package com.example.aquariumapp

import android.content.Intent
import android.graphics.Color
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import com.android.volley.Request
import com.android.volley.toolbox.JsonObjectRequest
import com.android.volley.toolbox.Volley
import com.example.aquariumapp.utils.EmptyResponseBodyRequest
import com.google.firebase.database.DataSnapshot
import com.google.firebase.database.DatabaseError
import com.google.firebase.database.FirebaseDatabase
import com.google.firebase.database.ValueEventListener
import com.google.gson.Gson
import org.json.JSONObject
import kotlin.properties.Delegates

data class Aquarium(
    var name:String,
    var ssid:String,
    var localIP:String,
    var manualMode:Boolean,
    var brightness:Int,
    var ph: Float,
    var phMin: Float,
    var phMax: Float,
    var temperature: Float,
    var temperatureMin: Float,
    var temperatureMax: Float
) {
    constructor() : this("", "", "", true, 0, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f)
}

class AquariumDetails : AppCompatActivity() {
    lateinit var id:String
    lateinit var name:String
    lateinit var localIP:String
    var remote by Delegates.notNull<Boolean>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_aquarium_details)

        id = intent.getStringExtra("id")!!
        remote = intent.getBooleanExtra("remote", false)
        localIP = intent.getStringExtra("localIP")!!

        if (remote) { // Remote calls
            val database = FirebaseDatabase.getInstance()
            val ref = database.getReference(id)

            val postListener = object : ValueEventListener {
                override fun onDataChange(dataSnapshot: DataSnapshot) {
                    val aquarium = dataSnapshot.getValue(Aquarium::class.java)!!
                    setDetails(aquarium)
                }

                override fun onCancelled(databaseError: DatabaseError) {
                    Log.w("warning", "loadPost:onCancelled", databaseError.toException())
                }
            }
            ref.addValueEventListener(postListener)

            findViewById<TextView>(R.id.decreaseBrightness).setOnClickListener {
                val values = HashMap<String, Int>()
                values["brightness"] = findViewById<TextView>(R.id.brightness).text.toString().toInt() - 1
                ref.updateChildren(values as Map<String, Any>)
            }
            findViewById<TextView>(R.id.increaseBrightness).setOnClickListener {
                val values = HashMap<String, Int>()
                values["brightness"] = findViewById<TextView>(R.id.brightness).text.toString().toInt() + 1
                ref.updateChildren(values as Map<String, Any>)
            }
            findViewById<TextView>(R.id.autoMode).setOnClickListener {
                val values = HashMap<String, Boolean>()
                values["manualMode"] = false
                ref.updateChildren(values as Map<String, Any>)
            }
            findViewById<TextView>(R.id.manualMode).setOnClickListener {
                val values = HashMap<String, Boolean>()
                values["manualMode"] = true
                ref.updateChildren(values as Map<String, Any>)
            }
        } else { // Local calls
            val queue = Volley.newRequestQueue(this)
            val handler = Handler(Looper.getMainLooper())
            handler.post(object: Runnable {
                override fun run() {
                    queue.add(JsonObjectRequest(
                        "http://$localIP",
                        { response ->
                            val aquarium = Gson().fromJson(response.toString(), Aquarium::class.java)
                            setDetails(aquarium)
                        },
                        { error -> Log.e("error", "Get information error: ", error) }
                    ))
                    handler.postDelayed(this, 1000)
                }
            })

            findViewById<TextView>(R.id.decreaseBrightness).setOnClickListener {
                val values = HashMap<String, Int>()
                values["brightness"] = findViewById<TextView>(R.id.brightness).text.toString().toInt() - 1
                queue.add(EmptyResponseBodyRequest(
                    Request.Method.PUT, "http://$localIP/brightness/decrease",
                    JSONObject(values as Map<String, Any>),
                    { _ -> Log.i("info", "ESP received the decrease brightness request") },
                    { Log.e("error", "Local request did not work") }
                ))
            }
            findViewById<TextView>(R.id.increaseBrightness).setOnClickListener {
                val values = HashMap<String, Int>()
                values["brightness"] = findViewById<TextView>(R.id.brightness).text.toString().toInt() + 1
                queue.add(EmptyResponseBodyRequest(
                    Request.Method.PUT, "http://$localIP/brightness/increase",
                    JSONObject(values as Map<String, Any>),
                    { _ -> Log.i("info", "ESP received the increase brightness request") },
                    { Log.e("error", "Local request did not work") }
                ))
            }
            findViewById<TextView>(R.id.autoMode).setOnClickListener {
                val values = HashMap<String, Boolean>()
                values["manualMode"] = false
                queue.add(EmptyResponseBodyRequest(
                    Request.Method.PUT, "http://$localIP/mode/auto",
                    JSONObject(values as Map<String, Any>),
                    { _ -> Log.i("info", "ESP received the auto mode request") },
                    { Log.e("error", "Local request did not work") }
                ))
            }
            findViewById<TextView>(R.id.manualMode).setOnClickListener {
                val values = HashMap<String, Boolean>()
                values["manualMode"] = true
                queue.add(EmptyResponseBodyRequest(
                    Request.Method.PUT, "http://$localIP/mode/manual",
                    JSONObject(values as Map<String, Any>),
                    { _ -> Log.i("info", "ESP received the auto mode request") },
                    { Log.e("error", "Local request did not work") }
                ))
            }
        }
    }

    fun setDetails(aquarium: Aquarium) {
        findViewById<TextView>(R.id.name).text = aquarium.name
        name = aquarium.name
        findViewById<TextView>(R.id.localOrRemote).text = if (remote) "Remote" else "Local"
        findViewById<TextView>(R.id.temperature).text = aquarium.temperature.toString()
        findViewById<TextView>(R.id.ph).text = aquarium.ph.toString()
        findViewById<TextView>(R.id.brightness).text = aquarium.brightness.toString()
        val decreaseBrightnessEnabled:Boolean = aquarium.brightness > 0 && aquarium.manualMode
        val increaseBrightnessEnabled:Boolean = aquarium.brightness < 4 && aquarium.manualMode
        findViewById<TextView>(R.id.decreaseBrightness).isEnabled = decreaseBrightnessEnabled
        findViewById<TextView>(R.id.decreaseBrightness).setBackgroundColor(if (decreaseBrightnessEnabled) Color.parseColor("#09846F") else Color.WHITE)
        findViewById<TextView>(R.id.decreaseBrightness).setTextColor(if (decreaseBrightnessEnabled) Color.WHITE else Color.BLACK)
        findViewById<TextView>(R.id.increaseBrightness).isEnabled = increaseBrightnessEnabled
        findViewById<TextView>(R.id.increaseBrightness).setBackgroundColor(if (increaseBrightnessEnabled) Color.parseColor("#09846F") else Color.WHITE)
        findViewById<TextView>(R.id.increaseBrightness).setTextColor(if (increaseBrightnessEnabled) Color.WHITE else Color.BLACK)
        findViewById<TextView>(R.id.autoMode).setBackgroundColor(if (!aquarium.manualMode) Color.parseColor("#09846F") else Color.parseColor("#DDDDDD"))
        findViewById<TextView>(R.id.manualMode).setBackgroundColor(if (aquarium.manualMode) Color.parseColor("#09846F") else Color.parseColor("#DDDDDD"))
        val temperatureDanger = if (aquarium.temperature < aquarium.temperatureMin || aquarium.temperature > aquarium.temperatureMax) View.VISIBLE else View.INVISIBLE
        findViewById<ImageView>(R.id.temperatureDanger).visibility = temperatureDanger
        val phDanger = if (aquarium.ph < aquarium.phMin || aquarium.ph > aquarium.phMax) View.VISIBLE else View.INVISIBLE
        findViewById<ImageView>(R.id.phDanger).visibility = phDanger
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        if (item.itemId == R.id.settings) {
            val intent = Intent(this, AquariumRanges::class.java)
            intent.putExtra("id", id)
            intent.putExtra("name", name)
            intent.putExtra("remote", remote)
            intent.putExtra("localIP", localIP)
            startActivity(intent)
        }
        return super.onOptionsItemSelected(item)
    }

    override fun onCreateOptionsMenu(menu: Menu?): Boolean {
        menuInflater.inflate(R.menu.aquarium_details, menu)
        return super.onCreateOptionsMenu(menu)
    }
}