package com.example.aquariumapp

import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.widget.Button
import android.widget.EditText
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import com.android.volley.*
import com.android.volley.toolbox.JsonObjectRequest
import com.android.volley.toolbox.Volley
import com.example.aquariumapp.utils.EmptyResponseBodyRequest
import com.google.firebase.database.DatabaseReference
import com.google.firebase.database.FirebaseDatabase
import org.json.JSONObject
import kotlin.properties.Delegates


private const val TEMPERATURE_MIN = "temperatureMin"
private const val TEMPERATURE_MAX = "temperatureMax"
private const val PH_MIN = "phMin"
private const val PH_MAX = "phMax"

class AquariumRanges : AppCompatActivity() {
    lateinit var id: String
    lateinit var localIP: String
    var remote by Delegates.notNull<Boolean>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_aquarium_ranges)

        id = intent.getStringExtra("id")!!
        findViewById<TextView>(R.id.name).text = intent.getStringExtra("name")
        localIP = intent.getStringExtra("localIP")!!
        remote = intent.getBooleanExtra("remote", false)
        findViewById<TextView>(R.id.localOrRemote).text = if (remote) "Remote" else "Local"

        if (remote) { // Remote calls
            val database = FirebaseDatabase.getInstance()
            val ref = database.reference.child(id)

            displayAquariumRangesRemotely(ref)
            setSaveCallbackRemotely(ref)
        } else { // Local calls
            val queue = Volley.newRequestQueue(this)
            displayAquariumRangesLocally(localIP, queue)
            setSaveCallbackLocally(localIP, queue)
        }

        /*findViewById<Button>(R.id.saveButton).setOnClickListener {
            val intent = Intent(this, AquariumDetails::class.java)
            intent.putExtra("id", id)
            intent.putExtra("remote", remote)
            intent.putExtra("localIP", localIP)
            startActivity(intent)
        }*/
    }

    override fun onBackPressed() {
        val intent = Intent(this, AquariumDetails::class.java)
        intent.putExtra("id", id)
        intent.putExtra("remote", remote)
        intent.putExtra("localIP", localIP)
        startActivity(intent)
        // super.onBackPressed()
    }

    private fun displayAquariumRangesRemotely(ref: DatabaseReference) {
        ref.get().addOnSuccessListener {
            Log.i("firebase", "Got value ${it.value}")
            // Update range views
            updateRanges(
                it.child(TEMPERATURE_MIN).value.toString().toDouble().toString(),
                it.child(TEMPERATURE_MAX).value.toString().toDouble().toString(),
                it.child(PH_MIN).value.toString().toDouble().toString(),
                it.child(PH_MAX).value.toString().toDouble().toString())
        }.addOnFailureListener {
            Log.e("firebase", "Error getting data", it)
        }
    }

    private fun displayAquariumRangesLocally(localIP: String, queue: RequestQueue) {
        val url = "http://$localIP"
        val request = JsonObjectRequest(
            url,
            { response ->
                // Update range views
                updateRanges(
                    response.get(TEMPERATURE_MIN).toString().toDouble().toString(),
                    response.get(TEMPERATURE_MAX).toString().toDouble().toString(),
                    response.get(PH_MIN).toString().toDouble().toString(),
                    response.get(PH_MAX).toString().toDouble().toString()
                )
            },
            { error -> Log.e("error", "Get ranges error: ", error) })

        queue.add(request)
    }

    private fun updateRanges(temperatureMin: String, temperatureMax: String,
                             phMin: String, phMax: String) {
        getCorrespondentView(TEMPERATURE_MIN).setText(temperatureMin)
        getCorrespondentView(TEMPERATURE_MAX).setText(temperatureMax)
        getCorrespondentView(PH_MIN).setText(phMin)
        getCorrespondentView(PH_MAX).setText(phMax)
    }

    private fun setSaveCallbackRemotely(ref: DatabaseReference) {
        val saveButton = findViewById<Button>(R.id.saveButton)
        saveButton.setOnClickListener {
            val ranges = HashMap<String, Double>()
            ranges[TEMPERATURE_MIN] = findViewById<EditText>(R.id.temperatureMin).text.toString().toDouble()
            ranges[TEMPERATURE_MAX] = findViewById<EditText>(R.id.temperatureMax).text.toString().toDouble()
            ranges[PH_MIN] = findViewById<EditText>(R.id.phMin).text.toString().toDouble()
            ranges[PH_MAX] = findViewById<EditText>(R.id.phMax).text.toString().toDouble()

            ref.updateChildren(ranges as Map<String, Any>)
            goBackToDetails()
        }
    }

    private fun setSaveCallbackLocally(localIP: String, queue: RequestQueue) {
        val url = "http://$localIP/ranges"
        val saveButton = findViewById<Button>(R.id.saveButton)
        saveButton.setOnClickListener {
            val ranges = HashMap<String, Double>()
            ranges[TEMPERATURE_MIN] = findViewById<EditText>(R.id.temperatureMin).text.toString().toDouble()
            ranges[TEMPERATURE_MAX] = findViewById<EditText>(R.id.temperatureMax).text.toString().toDouble()
            ranges[PH_MIN] = findViewById<EditText>(R.id.phMin).text.toString().toDouble()
            ranges[PH_MAX] = findViewById<EditText>(R.id.phMax).text.toString().toDouble()

            val request = EmptyResponseBodyRequest(
                Request.Method.PUT, url,
                JSONObject(ranges as Map<String, Double>),
                { _ ->
                    Log.i("info", "ESP32 received the new ranges")
                    goBackToDetails()
                },
                { error -> Log.e("error", "Send ranges error: ", error) }
            )
            queue.add(request)
        }
    }

    private fun goBackToDetails() {
        val intent = Intent(this, AquariumDetails::class.java)
        intent.putExtra("id", id)
        intent.putExtra("remote", remote)
        intent.putExtra("localIP", localIP)
        startActivity(intent)
    }

    private fun getCorrespondentView(property: String): EditText {
        val viewId = when(property) {
            TEMPERATURE_MIN -> R.id.temperatureMin
            TEMPERATURE_MAX -> R.id.temperatureMax
            PH_MIN -> R.id.phMin
            PH_MAX -> R.id.phMax
            else -> 0
        }

        return findViewById(viewId)
    }
}