#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <AsyncTCP.h>
#include <HTTPClient.h>
#include <ArduinoJson.h>
#include <AsyncJson.h>
#include <FirebaseESP32.h>

// Load Wi-Fi ssid and password credentials
#include "secrets.h"

class IntValue {
    public:
        int value = 0;
        bool valid = false;

        void constructor() {}

        int getValue() {return this->value;}
        void setValue(int value) {
            this->valid = true;
            this->value = value;
        }
        bool isValid() {return this->valid;}
        void setInvalid() {this->valid = false;}
};

class BoolValue {
    public:
        bool value = false;
        bool valid = false;

        void constructor() {}

        bool getValue() {return this->value;}
        void setValue(bool value) {
            this->valid = true;
            this->value = value;
        }
        bool isValid() {return this->valid;}
        void setInvalid() {this->valid = false;}
};

class DoubleValue {
    public:
        double value = 0.0;
        bool valid = false;

        void constructor() {}

        double getValue() {return this->value;}
        void setValue(double value){
            this->valid = true;
            this->value = value;
        }
        bool isValid() {return this->valid;}
        void setInvalid() {this->valid = false;}
};

// Define the Firebase Data object
FirebaseData fbdo;

// Define the FirebaseAuth data for authentication data
FirebaseAuth auth;

// Define the FirebaseConfig data for config data
FirebaseConfig config;

const int whiteLedPins[] = {15, 4, 5, 19};
const int whiteLedsNumber = sizeof(whiteLedPins) / sizeof(int);

const int redLedPin = 23;
const int yellowLedPin = 13;

const int temperaturePin = 34;
const int phPin = 35;

const int photoresistorPin = 32;

const String aquariumName = "Atlantis";

// Set web server port number to 80
AsyncWebServer server(80);

// Id of the aquarium
String id;

// How many white leds are on
IntValue brightness[4]; // remote, local, current, last

// Auto mode or manual mode for the brightness
BoolValue manualMode[4]; // remote, local, current, last

// Temperature level
double lastTemperature = 0.0;
double temperature = 0.0;
DoubleValue temperatureMin[4]; // remote, local, current, last
DoubleValue temperatureMax[4]; // remote, local, current, last
// double temperatureMinValue = 20.0;
// double temperatureMaxValue = 30.0;

// PH level
double lastPh = 0.0;
double ph = 0.0;
DoubleValue phMin[4]; // remote, local, current, last
DoubleValue phMax[4]; // remote, local, current, last
// double phMinValue = 6.0;
// double phMaxValue = 8.0;

// Time to update characteristics
unsigned long updateTime = millis();

bool signupOK = false;

// Flag to tell if ESP32 is getting the updates from Firebase
bool gettingFirebaseUpdates = false;

void initWifi()
{
    // Connect to Wi - Fi network with SSID and password
    Serial.print("Connecting to ");
    Serial.println(WIFI_SSID);
    WiFi.begin(WIFI_SSID, WIFI_PASSWORD);
    while (WiFi.status() != WL_CONNECTED)
    {
        delay(500);
        Serial.print(".");
    }
    // Print local IP address and start web server
    Serial.println("");
    Serial.println("WiFi connected.");
    Serial.print("IP address: ");
    Serial.println(WiFi.localIP());
}

void initFirebase()
{
      /* Assign the api key (required) */
    config.api_key = API_KEY;

    /* Assign the RTDB URL (required) */
    config.database_url = DATABASE_URL;

    /* Sign up */
    if (Firebase.signUp(&config, &auth, "", "")) {
        Serial.println("Signed up to Firebase");
        signupOK = true;
    }
    else {
        Serial.printf("%s\n", config.signer.signupError.message.c_str());
    }
    
    Firebase.begin(&config, &auth);
    Firebase.reconnectWiFi(true);
}

void setupServer()
{
    server.on("/", HTTP_GET, [](AsyncWebServerRequest *request) {
        StaticJsonDocument<200> doc;
        doc["name"] = aquariumName;
        doc["manualMode"] = manualMode[2].getValue();
        doc["brightness"] = brightness[2].getValue();
        doc["temperature"] = temperature;
        doc["temperatureMin"] = temperatureMin[2].getValue();
        doc["temperatureMax"] = temperatureMax[2].getValue();
        doc["ph"] = ph;
        doc["phMin"] = phMin[2].getValue();
        doc["phMax"] = phMax[2].getValue();

        String jsonString;
        serializeJson(doc, jsonString);
        request->send(200, "application/json", jsonString);
    });

    server.on("/brightness/increase", HTTP_PUT, [](AsyncWebServerRequest *request) {
        if (manualMode[2].getValue() && brightness[2].getValue() < whiteLedsNumber)
        {
            brightness[1].setValue(brightness[2].getValue() + 1);
            Serial.println("Increased brightness");
            request->send(200);
        } 
        else 
        {
            request->send(403);
        }
    });

    server.on("/brightness/decrease", HTTP_PUT, [](AsyncWebServerRequest *request) {
        if (manualMode[2].getValue() && brightness[2].getValue() > 0)
        {
            brightness[1].setValue(brightness[2].getValue() - 1);
            Serial.println("Decreased brightness");
            request->send(200);
        } 
        else 
        {
            request->send(403);
        }
    });

    server.on("/mode/auto", HTTP_PUT, [](AsyncWebServerRequest *request) {
        manualMode[1].setValue(false);
        Serial.println("Auto mode activated");
        request->send(200);
    });

    server.on("/mode/manual", HTTP_PUT, [](AsyncWebServerRequest *request) {
        manualMode[1].setValue(true);
        Serial.println("Manual mode activated");
        request->send(200);
    });

    server.on(
        "/ranges", 
        HTTP_PUT, 
        [](AsyncWebServerRequest *request){},
        NULL, [](AsyncWebServerRequest * request, uint8_t *data, size_t len, size_t index, size_t total){
            char* jsonString = (char*) data;
            
            DynamicJsonDocument doc(1024);
            deserializeJson(doc, jsonString);
            temperatureMin[1].setValue(doc["temperatureMin"]);
            temperatureMax[1].setValue(doc["temperatureMax"]);
            phMin[1].setValue(doc["phMin"]);
            phMax[1].setValue(doc["phMax"]);

            request->send(200);
    });

    server.begin();
}

void createAquarium()
{
    // Check WiFi connection and Firebase status
    if (WiFi.status() == WL_CONNECTED && Firebase.ready() && signupOK)
    {
        FirebaseJson json;

        json.set("/name", aquariumName);
        json.set("/temperature", temperature);
        json.set("/temperatureMin", temperatureMin[2].getValue());
        json.set("/temperatureMax", temperatureMax[2].getValue());
        json.set("/ph", ph);
        json.set("/phMin", phMin[2].getValue());
        json.set("/phMax", phMax[2].getValue());
        json.set("/manualMode", manualMode[2].getValue() ? true : false);
        json.set("/brightness", brightness[2].getValue());
        json.set("/ssid", String(WIFI_SSID));
        json.set("/localIP", WiFi.localIP().toString());

        Firebase.pushJSON(fbdo, "/", json);
        id = fbdo.pushName(); // Get id of the aquarium
    }
}

void setup()
{
    Serial.begin(115200);

    // Default values
    ph = 7.0;
    phMin[2].setValue(6.0);
    phMax[2].setValue(8.0);
    temperature = 25.0;
    temperatureMin[2].setValue(20.0);
    temperatureMax[2].setValue(30.0);
    brightness[2].setValue(0);
    manualMode[2].setValue(false);

    for (int i = 0; i < whiteLedsNumber; i++)
    {
        pinMode(whiteLedPins[i], OUTPUT);
        digitalWrite(whiteLedPins[i], LOW);
    }

    pinMode(redLedPin, OUTPUT);
    digitalWrite(redLedPin, LOW);

    pinMode(yellowLedPin, OUTPUT);
    digitalWrite(yellowLedPin, LOW);

    initWifi();
    initFirebase();
    setupServer();
    createAquarium();
}

int manualTune(int pin, int minMap, int maxMap, int min, int max)
{
    int value = analogRead(pin);
    value = map(value, minMap, maxMap, min, max);
    return constrain(value, min, max);
}

void photoresistor()
{
    int lightValue = manualTune(photoresistorPin, 0, 1023, 0, whiteLedsNumber);
    // Serial.println(lightValue);
    brightness[2].setValue(whiteLedsNumber - lightValue);
}

void manageWhiteLeds() {
    for (int i = 0; i < whiteLedsNumber; i++)
    {
        if (i < brightness[2].getValue())
            digitalWrite(whiteLedPins[i], HIGH);
        else
            digitalWrite(whiteLedPins[i], LOW);
    }
}

void checkTemperature()
{
    temperature = (double) manualTune(temperaturePin, 0, 4095, 0, 50);
    // Serial.println(temperature);
    if (temperature < temperatureMin[2].getValue() || temperature > temperatureMax[2].getValue())
    {
        digitalWrite(redLedPin, HIGH);
    }
}

void checkPH()
{
    ph = (double) manualTune(phPin, 0, 4095, 0, 14);
    // Serial.println(ph);
    if (ph < phMin[2].getValue() || ph > phMax[2].getValue())
    {
        digitalWrite(yellowLedPin, HIGH);
    }
}

// Obtain value of temperature and PH and check if
// they are out-of-bounds
void checkCharacteristics()
{
    checkTemperature();
    checkPH();
    delay(250);
    digitalWrite(redLedPin, LOW);
    digitalWrite(yellowLedPin, LOW);
}

void getUpdates()
{
    // Check WiFi connection and Firebase status
    if (WiFi.status() == WL_CONNECTED && Firebase.ready() && signupOK)
    {
        if (Firebase.getJSON(fbdo, "/" + id)) {
            FirebaseJson json = fbdo.jsonObject();
            FirebaseJsonData result;

            json.get(result, "manualMode");
            bool manualModeValue = result.to<bool>();
            if (manualModeValue != manualMode[0].getValue())
                manualMode[0].setValue(manualModeValue);

            if (manualMode[0].getValue()) {
                json.get(result, "brightness");
                int brightnessValue = result.to<int>();
                if (brightnessValue != brightness[0].getValue())
                    brightness[0].setValue(brightnessValue);
            }

            json.get(result, "temperatureMin");
            double temperatureMinValue = result.to<double>();
            if (temperatureMinValue != temperatureMin[0].getValue())
                temperatureMin[0].setValue(temperatureMinValue);
            json.get(result, "temperatureMax");
            double temperatureMaxValue = result.to<double>();
            if (temperatureMaxValue != temperatureMax[0].getValue())
                temperatureMax[0].setValue(temperatureMaxValue);

            json.get(result, "phMin");
            double phMinValue = result.to<double>();
            if (phMinValue != phMin[0].getValue())
                phMin[0].setValue(phMinValue);
            json.get(result, "phMax");
            double phMaxValue = result.to<double>();
            if (phMaxValue != phMax[0].getValue())
                phMax[0].setValue(phMaxValue);
        }
        } else { // Failed to get JSON data at defined database path, print out the error reason
            Serial.println(fbdo.errorReason());
        }
}

void updateCharacteristics()
{
    FirebaseJson json;

    if (lastTemperature != temperature)
        json.set("/temperature", temperature);
    if (lastPh != ph)
        json.set("/ph", ph);
    if (brightness[2].getValue() != brightness[3].getValue())
        json.set("/brightness", brightness[2].getValue());
    if (manualMode[2].getValue() != manualMode[3].getValue())
        json.set("/manualMode", manualMode[2].getValue());
    if (temperatureMin[2].getValue() != temperatureMin[3].getValue())
        json.set("/temperatureMin", temperatureMin[2].getValue());
    if (temperatureMax[2].getValue() != temperatureMax[3].getValue())
        json.set("/temperatureMax", temperatureMax[2].getValue());
    if (phMin[2].getValue() != phMin[3].getValue())
        json.set("/phMin", phMin[2].getValue());
    if (phMax[2].getValue() != phMax[3].getValue())
        json.set("/phMax", phMax[2].getValue());

    // if (changedRanges) {
    //     json.set("/temperatureMin", temperatureMinValue);
    //     json.set("/temperatureMax", temperatureMaxValue);
    //     json.set("/phMin", phMinValue);
    //     json.set("/phMax", phMaxValue);
    //     changedRanges = false;
    // }

    Firebase.updateNodeSilent(fbdo, "/" + id, json);
}

boolean characteristicsChanged()
{
    return brightness[2].getValue() != brightness[3].getValue() 
        || manualMode[2].getValue() != manualMode[3].getValue() 
        || lastTemperature != temperature
        || lastPh != ph
        || temperatureMin[2].getValue() != temperatureMin[3].getValue()
        || temperatureMax[2].getValue() != temperatureMax[3].getValue()
        || phMin[2].getValue() != phMin[3].getValue()
        || phMax[2].getValue() != phMax[3].getValue();
}

template <typename T>
void updateCurrentValue(T* currentValue, T* remoteValue, T* localValue) {
    if (remoteValue->isValid())
        currentValue->setValue(remoteValue->getValue());
    if (localValue->isValid())
        currentValue->setValue(localValue->getValue());
    remoteValue->setInvalid();
    localValue->setInvalid();
}

void loop()
{
    // Get remote updates
    getUpdates();

    // Updates old values
    brightness[3].setValue(brightness[2].getValue());
    manualMode[3].setValue(manualMode[2].getValue());
    lastTemperature = temperature;
    temperatureMin[3].setValue(temperatureMin[2].getValue());
    temperatureMax[3].setValue(temperatureMax[2].getValue());
    lastPh = ph;
    phMin[3].setValue(phMin[2].getValue());
    phMax[3].setValue(phMax[2].getValue());

    // Update current from remote and local
    updateCurrentValue(&brightness[2], &brightness[0], &brightness[1]);
    updateCurrentValue(&manualMode[2], &manualMode[0], &manualMode[1]);
    updateCurrentValue(&temperatureMin[2], &temperatureMin[0], &temperatureMin[1]);
    updateCurrentValue(&temperatureMax[2], &temperatureMax[0], &temperatureMax[1]);
    updateCurrentValue(&phMin[2], &phMin[0], &phMin[1]);
    updateCurrentValue(&phMax[2], &phMax[0], &phMax[1]);

    // Update current for temperature, ph and brightness
    checkCharacteristics();
    if (!manualMode[2].getValue())
        photoresistor();

    if (characteristicsChanged())
    {
        Serial.println("Updating characteristics!");
        updateCharacteristics();
    }

    manageWhiteLeds();
    delay(1000);
}
