/**
 * @author Ricardo Cruz Nr 34951
 * @author Ricardo Gaspar Nr 42038
 * @author Luís Silva Nr 34535
 * Docente: Francisco Azevedo	P4
 */
package circuit;

import java.util.Random;

import org.jfree.data.xy.XYSeries;

/**
 * Classe que "implementa" o algoritmo genético
 */
public class GeneticAlgorithm {

	private Population population;
	private Random rand;
	private double lastPopulationFitness;
	private double lastBestFitEnough;

	private double lastBestFit;
	private double lastWorstFit;
	private long bestFitFirstAppearanceGeneration = 1;
	private long worstFitFirstAppearanceGeneration = 1;

	public static final int DEFAULT_INITIAL_POPULATION_SIZE = 6;

	// Probabilidades
	public static final float DEFAULT_CROSSOVER_PROBABLITY = 20f; // 20%
	public static final float DEFAULT_MUTATION_PROBABLITY = 20f; // 20%

	private static final float MAX_RANDOM_FLOAT = 100f;
	private float crossoverProbability;
	private float mutationProbability;

	// Variáveis para os critérios de paragem
	private int enoughCounter = 1;
	private static final int MAX_IS_FIT_ENOUGH = 5;
	private String stopCriteria;
	private long stopCriteriaValue;
	private int numberOfGenerations = 1;
	private long startTime;
	private long elapsedTime;

	// Critérios de paragem
	public static final String NO_OF_GENERATIONS_LIMIT = "GENERATIONS LIMIT";
	public static final String TIME_LIMIT = "TIME LIMIT";
	public static final String INDIVIDUAL_IS_FIT_ENOUGH = "BEST FIT STABBILIZATION";
	public static final String POPULATION_IS_FIT_ENOUGH = "POPULATION STABILIZATION";

	public static final int DEFAULT_STOP_CRITERIA_GENERATION_VALUE = 5;
	public static final int DEFAULT_STOP_CRITERIA_TIME_VALUE = 1000;

	// Controlo do elitismo
	public static final int DEFAULT_ELITE_POPULATION_SIZE = 2;
	private boolean isElitismOn = false;
	private int eliteSize = DEFAULT_ELITE_POPULATION_SIZE;

	// Dados para o gráfico
	private XYSeries chartSeries;

	/**
	 * Construtor
	 * 
	 * @param pop
	 *            uma população
	 */
	GeneticAlgorithm(Population pop) {
		population = pop;
		rand = new Random();
		crossoverProbability = DEFAULT_CROSSOVER_PROBABLITY;
		mutationProbability = DEFAULT_MUTATION_PROBABLITY;

		setStopCriteria(INDIVIDUAL_IS_FIT_ENOUGH, 0); 
		setElitism(true, 0);
	}

	/**
	 * Construtor
	 * 
	 * @param pop
	 *            uma população
	 * @param pcrossover
	 *            a probabilidade de crossover
	 * @param pmutate
	 *            a probabilidade de mutação
	 */
	GeneticAlgorithm(Population pop, float pcrossover, float pmutate) {
		population = pop;
		rand = new Random();
		crossoverProbability = pcrossover;
		mutationProbability = pmutate;

		setStopCriteria(INDIVIDUAL_IS_FIT_ENOUGH, 0);
		setElitism(true, 0);
	}

	/**
	 * Método que pesquisa e devolve o melhor indivíduo encontrado
	 * 
	 * @return pop.getBestIndividual(), o melhor indivíduo
	 */
	public Individual search() {
		// Variáveis dos critérios de paragem
		numberOfGenerations = 1;
		startTime = System.currentTimeMillis();
		enoughCounter = 1;

		// Aparições do melhor e pior
		bestFitFirstAppearanceGeneration = 1;
		worstFitFirstAppearanceGeneration = 1;
		lastBestFit = population.getBestIndividual().fitness();
		lastWorstFit = population.getWorstIndividual().fitness();

		Population newPopulation;
		Individual[] children = new Individual[2];
		int halfSize = population.size / 2;

		do {
			chartSeries.add(numberOfGenerations, population.getBestIndividual()
					.fitness()); //dados para o gráfico
			
			if (!isElitismOn) {
				newPopulation = new Population();
			} else {
				newPopulation = copyElite(population);
				halfSize = (population.size - eliteSize) / 2;

			}

			for (int i = 0; i < halfSize; i++) {
				RoverCircuit x = (RoverCircuit) population.selectIndividual();
				RoverCircuit y = (RoverCircuit) population.selectIndividual();
				if (nextFloat() <= crossoverProbability) {
					children = x.crossover(y);
				} else {
					children[0] = x;
					children[1] = y;
				}

				if (nextFloat() <= mutationProbability)
					children[0].mutate();
				if (nextFloat() <= mutationProbability)
					children[1].mutate();

				newPopulation.addIndividual(children[0]);
				newPopulation.addIndividual(children[1]);
				
			}
			population = newPopulation;

			// ver o pior individuo
			if (population.getWorstIndividual().fitness() > lastWorstFit) {
				lastWorstFit = population.getWorstIndividual().fitness();
				worstFitFirstAppearanceGeneration = numberOfGenerations;
			}
			// ver o melhor
			if (!isElitismOn) {
				bestFitFirstAppearanceGeneration = numberOfGenerations;
			} else {
				if (population.getBestIndividual().fitness() < lastBestFit) {
					bestFitFirstAppearanceGeneration = numberOfGenerations;
				}

				lastBestFit = population.getBestIndividual().fitness();

			}

			numberOfGenerations++;
			elapsedTime = System.currentTimeMillis() - startTime;

		} while (!stopCriteriaReached(population));
		
		return population.getBestIndividual();

	}

	/**
	 * Devolve em que geração apareceu o melhor indivíduo. No caso da procura
	 * sem elitismo, é a última geração.
	 * 
	 * @return que geração apareceu o melhor indivíduo.
	 */
	public long getBestFitFirstAppearance() {
		return bestFitFirstAppearanceGeneration;
	}

	/**
	 * Devolve em que geração apareceu o pior indivíduo.
	 * 
	 * @return que geração apareceu o pior indivíduo.
	 */
	public long getWorstFitFirstAppearance() {
		return worstFitFirstAppearanceGeneration;
	}

	/**
	 * Devolve o fitness do pior indivíduo.
	 * 
	 * @return o fitness do pior indivíduo.
	 */
	public double getWorstFitness() {
		return lastWorstFit;
	}

	/**
	 * Devolve o número de gerações geradas após a pesquisa.
	 * 
	 * @return número de gerações.
	 */
	public long getNumberOfGenerations() {
		return numberOfGenerations;
	}

	/**
	 * Devolve o tempo decorrido desde o início da pesquisa.
	 * 
	 * @return tempo decorrido.
	 */
	public long getElapsedTime() {
		return elapsedTime;
	}

	/**
	 * Define o critério de paragem e o seu valor.
	 * 
	 * @param stopCriteria
	 *            critério de paragem. <br>
	 *            Opções possíveis: <br>
	 *            <blockquote> - Número de Gerações = NO_OF_GENERATIONS_LIMIT <br>
	 *            - Limite de tempo (em milissegundos) = TIME_LIMIT <br>
	 *            - Estabilização do valor de fitness do melhor indivíduo =
	 *            INDIVIDUAL_IS_FIT_ENOUGH <br>
	 *            - Estabilização do valor de fitness da melhor população =
	 *            POPULATION_IS_FIT_ENOUGH <br>
	 *            </blockquote>
	 * @param value
	 *            valor do critério de paragem. <br>
	 *            Interpretação dos valores para os critérios de paragem:<br>
	 *            <blockquote> - Número de Gerações -> número de gerações máximo
	 *            a gerar. <br>
	 *            - Limite de tempo -> tempo máximo de execução (em
	 *            milissegundos). <br>
	 *            - Estabilização do valor de fitness do melhor indivíduo -> não
	 *            necessita de valor de paragem, qualquer valor é aceite.<br>
	 *            - Estabilização do valor de fitness da melhor população -> não
	 *            necessita de valor de paragem, qualquer valor é aceite.<br>
	 *            </blockquote>
	 */
	public void setStopCriteria(String stopCriteria, long value) {
		this.stopCriteria = stopCriteria;
		this.stopCriteriaValue = value;
	}

	/**
	 * Altera o estado do filtro de elitismo.
	 * 
	 * @param state
	 *            estado do elitismo.
	 * @param eliteSize
	 *            valor do elitismo.
	 */
	public void setElitism(boolean state, int eliteSize) {
		isElitismOn = state;
		this.eliteSize = eliteSize;
	}

	/**
	 * Afecta o objecto para a inserção de dados no gráfico.
	 * 
	 * @param series
	 *            objecto para se inserir os valores no gráfico.
	 */
	public void setChartData(XYSeries series) {
		this.chartSeries = series;

	}

	/**
	 * Verifica se o critério de paragem já foi atingido.
	 * 
	 * @return <code>true</code> quando o critério de paragem é atindo.
	 *         <code>false</code> caso contrário.
	 */
	private boolean stopCriteriaReached(Population population) {
		boolean result = false;
		switch (stopCriteria) {
		case NO_OF_GENERATIONS_LIMIT:
			result = numberOfGenerations == stopCriteriaValue;
			break;
		case TIME_LIMIT:
			result = elapsedTime >= stopCriteriaValue;
			break;
		case INDIVIDUAL_IS_FIT_ENOUGH:
			result = isIndividualFitEnough(population);
			break;
		case POPULATION_IS_FIT_ENOUGH:
			result = isPopulationFitEnough(population);
			break;

		default:
			break;
		}

		return result;
	}

	/**
	 * Gera um número aleatório (de 1 a 100) para comparar com as
	 * probabilidades. Começa em 1 para que quando as probabildades de crossover
	 * e mutação forem 0, não efectuar nenhuma destas operações. Visto que o
	 * algoritmo especifica que o oprador de comparação entre estes valores é o
	 * <=.
	 * 
	 * @return valor float aleatório de 1 a 100.
	 */
	private float nextFloat() {
		float result = rand.nextFloat() * MAX_RANDOM_FLOAT;
		if (result == 0)
			result = 1;
		return result;
	}

	/**
	 * Indica se o valor de fitness do melhor indivíduo já estabilizou e não se
	 * altera durante MAX_IS_FIT_ENOUGH gerações.
	 * 
	 * @param population
	 *            população actual.
	 * @return <code>true</code> quando valor de fitness estabilzou. Caso
	 *         contrário devolve <code>false</code>.
	 */
	private boolean isIndividualFitEnough(Population population) {
		double currentPopulationBestFit = population.getBestIndividual()
				.fitness();

		if (lastBestFitEnough == currentPopulationBestFit)
			enoughCounter++;
		else {
			lastBestFitEnough = currentPopulationBestFit;
			enoughCounter = 1;
		}

		if (enoughCounter == MAX_IS_FIT_ENOUGH)
			return true;

		return false;
	}

	/**
	 * Indica se o valor total de fitness da melhor população já estabilizou e
	 * não se altera durante MAX_IS_FIT_ENOUGH gerações.
	 * 
	 * @param population
	 *            população actual.
	 * @return <code>true</code> quando valor de fitness estabilzou. Caso
	 *         contrário devolve <code>false</code>.
	 */
	private boolean isPopulationFitEnough(Population population) {
		double currentPopulationFitness = population.getTotalFitness();

		if (lastPopulationFitness == currentPopulationFitness)
			enoughCounter++;
		else {
			lastPopulationFitness = currentPopulationFitness;
			enoughCounter = 1;
		}

		if (enoughCounter == MAX_IS_FIT_ENOUGH)
			return true;

		return false;
	}

	/**
	 * Cria uma nova população com a elite.
	 * 
	 * @param pop
	 *            população antiga.
	 * @return uma nova população com a elite.
	 */
	private Population copyElite(Population pop) {
		return new Population(pop.getElite(eliteSize));
	}

}