/**
 * @author Ricardo Cruz Nr 34951
 * @author Ricardo Gaspar Nr 42038
 * @author Luís Silva Nr 34535
 * Docente: Francisco Azevedo	P4
 */
package circuit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Random;
import java.util.Collections;

/**
 * Classe usada para a representação de uma população.
 */
public class Population {

	private ArrayList<Individual> pop;
	private ArrayList<Double> acum;

	private boolean currupt;
	private double worstFit;
	private Individual worstInd;
	private double bestFit;
	private Individual bestInd;
	private double sumOfFitness;

	private Random gen;
	protected int size;

	/**
	 * Construtor relativo à classe Population
	 */
	public Population() {
		pop = new ArrayList<Individual>(0);
		gen = new Random();
		size = 0;

		currupt = true;
		acum = new ArrayList<Double>();
		updateOverallFitness();
	}

	/**
	 * Construtor onde se especifica a população.
	 * 
	 * @param p
	 *            um array de indivíduos
	 */
	public Population(Individual[] p) {
		pop = new ArrayList<Individual>(Arrays.asList(p));
		size = pop.size();
		gen = new Random();

		currupt = true;
		acum = new ArrayList<Double>();
		updateOverallFitness();

	}

	/**
	 * Selecciona e devolve um indivíduo da população, tendo em conta a sua
	 * fitness
	 * 
	 * @return um array de indivíduos
	 */
	public Individual selectIndividual() {
		// Verifica se necessita de calcular os valores de probabilidade de
		// selecção de cada indivíduo
		if (currupt) {
			double total = 0.0;
			for (int i = 0; i < pop.size(); i++) {
				total += 1 / pop.get(i).fitness();
				acum.add(total / sumOfFitness);
			}
			currupt = false;
		}

		double r = gen.nextDouble();

		Collections.sort(acum);
		int pos = Collections.binarySearch(acum, r);

		if (pos >= 0)
			return pop.get(pos);
		else if (-1 * (pos + 1) >= size) // Para evitar o erro da pos ser
											// superior à última posição da
											// populaçao
			return pop.get(size - 1);
		else
			return pop.get(-(pos + 1));

	}

	/**
	 * Adiciona um indivíduo à população
	 * 
	 * @param ind
	 *            , um indivíduo
	 */
	public void addIndividual(Individual ind) {
		size++;
		pop.add(ind);
		double f = ind.fitness();
		sumOfFitness += 1 / f;

		updateOverallFitness();
	}

	@Override
	public String toString() {
		StringBuilder individuals = new StringBuilder();
		for (int i = 0; i < size; i++) {
			individuals.append(pop.get(i).toString() + "\n");
		}
		return "Population: \n" + individuals;
	}

	/**
	 * Devolve o fitness total da população.
	 * 
	 * @return soma dos valores de fitness de todos os indivíduos da população.
	 */
	public double getTotalFitness() {
		double totalFitness = 0;
		for (int i = 0; i < size; i++) {
			totalFitness += pop.get(i).fitness();
		}
		return totalFitness;
	}

	/**
	 * Devolve o melhor indivíduo da população.
	 * 
	 * @return o indivíduo com maior valor de fitness.
	 */
	public Individual getBestIndividual() {
		return bestInd;
	}

	/**
	 * Devolve o pior indivíduo da população.
	 * 
	 * @return o indivíduo com menor valor de fitness.
	 */
	public Individual getWorstIndividual() {
		return worstInd;
	}

	/**
	 * Actualiza os valores de fitness (melhor e pior) da população.
	 */
	private void updateOverallFitness() {
		worstFit = -1;
		bestFit = Double.MAX_VALUE;
		sumOfFitness = 0;

		for (int i = 0; i < size; i++) {
			double f = pop.get(i).fitness();
			sumOfFitness += 1 / f;
			if (f > worstFit) {
				worstFit = f;
				worstInd = pop.get(i);
			}
			if (f < bestFit) {
				bestFit = f;
				bestInd = pop.get(i);
			}
		}
	}

	/**
	 * Procura a elite com a dimensão indicada e devolve-a.
	 * 
	 * @param eliteSize
	 *            dimensão da elite.
	 * @returna elite com a dimensão indicada.
	 */
	public Individual[] getElite(int eliteSize) {
		//Criar um arraylist ordenado
		ArrayList<Individual> sortedPop = (ArrayList<Individual>) pop.clone();

		Collections.sort(sortedPop, new Comparator<Individual>() {
			public int compare(Individual i1, Individual i2) {
				if (i1.fitness() > i2.fitness())
					return 1;
				else if (i1.fitness() < i2.fitness())
					return -1;
				else
					return 0;
			}
		});
		//Converter para array
		Individual[] sortedPopArray = new Individual[size];
		sortedPop.toArray(sortedPopArray);
		//Copiar para o array da elite
		Individual[] elite = new Individual[eliteSize];
		for (int i = 0; i < elite.length; i++) {
			elite[i] = (Individual) sortedPopArray[i].clone();

		}

		return elite;

	}
}
