package motion;

import searchalgorithm.Node;
import searchproblem.*;

public class RoverProblem extends InformedSearchProblem {

	int goalXPos, goalYPos, goalZPos;
	private static int STRAIGHT_LINE_COST = 10;
	private static double DIAGONAL_LINE_COST = Math.sqrt(200);

	public RoverProblem(State initial, State goal) {
		super(initial, goal);
		goalXPos = ((RoverState) goal).getCoordX();
		goalYPos = ((RoverState) goal).getCoordY();
		goalZPos = ((RoverState) goal).getHeight();
	}

	@Override
	public double heuristic(Node n) {
		RoverState rs = (RoverState) n.getState();

//		return euclideanDistanceXY(rs);
//		return euclideanDistanceXYZ(rs);
//		 return manhattanDistance(rs);
		 return diagonalDistance(rs);
//		  return diagonalDistanceStaight(rs);
		// return 0;
	}

	/**
	 * Euclidean Distance heuristic. Calculates the Euclidean distance based on
	 * the current position and the destination.
	 * 
	 * @param rs
	 *            State of the rover.
	 * @return The heuristic value.
	 */
	private double euclideanDistanceXY(RoverState rs) {
		double dx = Math.pow((rs.getCoordX() - goalXPos), 2);
		double dy = Math.pow((rs.getCoordY() - goalYPos), 2);

		return STRAIGHT_LINE_COST * Math.sqrt(dx + dy);
	}

	/**
	 * Euclidean Distance XYZ heuristic. Calculates the Euclidean distance based
	 * on the current position and the destination, accounting with the height.
	 * 
	 * @param rs
	 *            State of the rover.
	 * @return The heuristic value.
	 */
	private double euclideanDistanceXYZ(RoverState rs) {
		double dx = Math.pow((rs.getCoordX() - goalXPos), 2);
		double dy = Math.pow((rs.getCoordY() - goalYPos), 2);
		double dz = Math.pow((rs.getHeight() - goalZPos), 2);

		return STRAIGHT_LINE_COST * Math.sqrt(dx + dy + dz);
	}

	/**
	 * Manhattan Distance heuristic. Calculates the Manhattan distance based on
	 * the current position and the destination.
	 * 
	 * @param rs
	 *            State of the rover.
	 * @return The heuristic value.
	 */
	private double manhattanDistance(RoverState rs) {
		double dx = Math.abs(rs.getCoordX() - goalXPos);
		double dy = Math.abs(rs.getCoordY() - goalYPos);

		return STRAIGHT_LINE_COST * (dx + dy);
	}

	/**
	 * Diagonal Distance heuristic. Calculates the Diagonal distance based on
	 * the current position and the destination. This assumes that the straight
	 * line and the diagonal have the same cost.
	 * 
	 * @param rs
	 *            State of the rover.
	 * @return The heuristic value.
	 */
	private double diagonalDistanceStraight(RoverState rs) {
		double dx = Math.abs(rs.getCoordX() - goalXPos);
		double dy = Math.abs(rs.getCoordY() - goalYPos);

		return STRAIGHT_LINE_COST * Math.max(dx, dy);
	}

	/**
	 * Diagonal Distance heuristic. Calculates the Diagonal distance based on
	 * the current position and the destination.
	 * 
	 * @param rs
	 *            State of the rover.
	 * @return The heuristic value.
	 */
	private double diagonalDistance(RoverState rs) {
		double dx = Math.abs(rs.getCoordX() - goalXPos);
		double dy = Math.abs(rs.getCoordY() - goalYPos);

		return STRAIGHT_LINE_COST * (dx + dy) + (DIAGONAL_LINE_COST - 2 * STRAIGHT_LINE_COST) * Math.min(dx, dy);
	}

}
