package motion;

import java.util.ArrayList;
import java.util.List;

import motion.Terrain.TerrainType;
import searchproblem.*;

/**
 * @author Ricardo Cruz Nr 34951
 * @author Ricardo Gaspar Nr 42038
 * @author Lus Silva Nr 34535 Docente: Francisco Azevedo P4
 */
public class RoverState extends State implements Cloneable {
	private int xPos;
	private int yPos;
	private Terrain map;
	private int height;
	private static int STEP = 1;
	private static int MAX_HEIGHT_INCLINE = 10;
	private static int STRAIGHT_LINE_COST = 10;
	private static double DIAGONAL_LINE_COST = Math.sqrt(200);

	public enum RoverOperator {
		N, S, E, W, NE, SE, SW, NW
	}

	/**
	 * Initializes the rover state with the coordinates (0,0) on a given terrain
	 * map.
	 * 
	 * @param map
	 *            Map of the terrain.
	 */
	public RoverState(Terrain map) {
		xPos = 0;
		yPos = 0;
		this.map = map;
		height = map.getHeight(xPos, yPos);
	}

	/**
	 * Initializes the rover state with the given coordinates (x,y) on a given
	 * terrain map.
	 * 
	 * @param map
	 *            Map of the terrain.
	 * @param x
	 *            Coordinate x.
	 * @param y
	 *            Coordinate y.
	 */
	public RoverState(int x, int y, Terrain map) {
		xPos = x;
		yPos = y;
		this.map = map;
		height = map.getHeight(xPos, yPos);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#successorFunction()
	 */
	@Override
	public List<Arc> successorFunction() {
		List<Arc> children = new ArrayList<Arc>(8);
		for (RoverOperator op : RoverOperator.values()) {
			if (applicableOperator(op))
				children.add(successorState(op));
		}
		return children;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#successorState(java.lang.Object)
	 */
	@Override
	public Arc successorState(Object op) {
		RoverState child = (RoverState) this.clone();
		return new Arc(this, child, op, child.applyOperator(op));

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#applyOperator(java.lang.Object)
	 */
	@Override
	public double applyOperator(Object op) {
		RoverOperator operator = (RoverOperator) op;
		int deltaHeight;

		switch (operator) {
		case N:
			deltaHeight = map.getHeight(xPos, yPos - STEP)
					- map.getHeight(xPos, yPos);
			yPos -= STEP; // yy axis grows in south direction
			return getActionCost(xPos, yPos, deltaHeight) * STRAIGHT_LINE_COST;
		case S:
			deltaHeight = map.getHeight(xPos, yPos + STEP)
					- map.getHeight(xPos, yPos);
			yPos += STEP;
			return getActionCost(xPos, yPos, deltaHeight) * STRAIGHT_LINE_COST;
		case E:
			deltaHeight = map.getHeight(xPos + STEP, yPos)
					- map.getHeight(xPos, yPos);
			xPos += STEP;
			return getActionCost(xPos, yPos, deltaHeight) * STRAIGHT_LINE_COST;
		case W:
			deltaHeight = map.getHeight(xPos - STEP, yPos)
					- map.getHeight(xPos, yPos);
			xPos -= STEP;
			return getActionCost(xPos, yPos, deltaHeight) * STRAIGHT_LINE_COST;
		case NE:
			deltaHeight = map.getHeight(xPos + STEP, yPos - STEP)
					- map.getHeight(xPos, yPos);
			yPos -= STEP;
			xPos += STEP;
			return getActionCost(xPos, yPos, deltaHeight) * DIAGONAL_LINE_COST;
		case NW:
			deltaHeight = map.getHeight(xPos - STEP, yPos - STEP)
					- map.getHeight(xPos, yPos);
			yPos -= STEP;
			xPos -= STEP;
			return getActionCost(xPos, yPos, deltaHeight) * DIAGONAL_LINE_COST;
		case SE:
			deltaHeight = map.getHeight(xPos + STEP, yPos + STEP)
					- map.getHeight(xPos, yPos);
			yPos += STEP;
			xPos += STEP;
			return getActionCost(xPos, yPos, deltaHeight) * DIAGONAL_LINE_COST;
		case SW:
			deltaHeight = map.getHeight(xPos - STEP, yPos + STEP)
					- map.getHeight(xPos, yPos);
			yPos += STEP;
			xPos -= STEP;
			return getActionCost(xPos, yPos, deltaHeight) * DIAGONAL_LINE_COST;
		}
		return 0;
	}

	/**
	 * Calculates the cost of an action. This method takes into account the type
	 * of the terrain and the heights difference.
	 * 
	 * @param coordX
	 *            XX coordinate of the destination.
	 * @param coordY
	 *            YY coordinate of the destination.
	 * @param deltaHeight
	 *            difference of heights between the current position and the
	 *            destination position.
	 * @return the cost associated with the action taken.
	 */
	private double getActionCost(int coordX, int coordY, int deltaHeight) {
		TerrainType ttype = map.getTerrainType(coordX, coordY);
		switch (ttype) {
		case ROCK:
			return 3 * Math.exp(Math.sqrt(Math.abs(deltaHeight)));
		case SAND:
			return 2 * Math.exp(Math.sqrt(Math.abs(deltaHeight)));
		default:
			return 1 * Math.exp(Math.sqrt(Math.abs(deltaHeight)));
		}
	}

	/**
	 * Verifies if a given action is possible considering the current position.
	 * 
	 * @param action
	 *            action to perform.
	 * @return
	 */
	public boolean applicableOperator(Object action) {
		if (!(action instanceof RoverOperator))
			return false;
		RoverOperator op = (RoverOperator) action;
		switch (op) {
		case N:
			return yPos > 0
					&& (map.getHeight(xPos, yPos - STEP) - map.getHeight(xPos,
							yPos)) <= MAX_HEIGHT_INCLINE;
		case S:
			return yPos < map.getHorizontalSize() - 1
					&& (map.getHeight(xPos, yPos + STEP) - map.getHeight(xPos,
							yPos)) <= MAX_HEIGHT_INCLINE;
		case E:
			return xPos < map.getVerticalSize() - 1
					&& (map.getHeight(xPos + STEP, yPos) - map.getHeight(xPos,
							yPos)) <= MAX_HEIGHT_INCLINE;
		case W:
			return xPos > 0
					&& (map.getHeight(xPos - STEP, yPos) - map.getHeight(xPos,
							yPos)) <= MAX_HEIGHT_INCLINE;
		case NE:
			return yPos > 0
					&& xPos < map.getVerticalSize() - 1
					&& (map.getHeight(xPos + STEP, yPos - STEP) - map
							.getHeight(xPos, yPos)) <= MAX_HEIGHT_INCLINE;
		case NW:
			return yPos > 0
					&& xPos > 0
					&& (map.getHeight(xPos - STEP, yPos - STEP) - map
							.getHeight(xPos, yPos)) <= MAX_HEIGHT_INCLINE;
		case SE:
			return yPos < map.getHorizontalSize() - 1
					&& xPos < map.getVerticalSize() - 1
					&& (map.getHeight(xPos + STEP, yPos + STEP) - map
							.getHeight(xPos, yPos)) <= MAX_HEIGHT_INCLINE;
		case SW:
			return yPos < map.getHorizontalSize() - 1
					&& xPos > 0
					&& (map.getHeight(xPos - STEP, yPos + STEP) - map
							.getHeight(xPos, yPos)) <= MAX_HEIGHT_INCLINE;
		default:
			return false;
		}
	}

	/**
	 * Returns the current X coordinate.
	 * 
	 * @return X coordinate.
	 */
	public int getCoordX() {
		return xPos;
	}

	/**
	 * Returns the current Y coordinate.
	 * 
	 * @return Y coordinate.
	 */
	public int getCoordY() {
		return yPos;
	}

	/**
	 * Returns the height of the current position.
	 * 
	 * @return The height of the current position.
	 */
	public int getHeight() {
		return height;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#clone()
	 */
	@Override
	public Object clone() {
		return new RoverState(xPos, yPos, map);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#hashCode()
	 */
	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + xPos;
		result = PRIME * result + yPos;
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see searchproblem.State#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final RoverState other = (RoverState) obj;
		if (this.xPos != other.xPos || this.yPos != other.yPos)
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "rover position (" + xPos + "," + yPos + ","+ height +")";
	}
}
