/**
 * 
 */
package schoolSystem;

import java.io.Serializable;

import dataStructures.*;

/**
 * @author Ricardo Gaspar Nr. 35277
 * @author Hugo Antnio Nr. 34334 Turno P2 Docente Vasco Amaral
 * 
 *         Classe que implementa a interface Card.
 */
public class CardClass implements CardWritable, Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Cdigo identificador do carto.
	 */
	private int cardCode;

	/**
	 * Nome do aluno.
	 */
	private String studentName;

	/**
	 * Turma do aluno.
	 */
	private String studentClass;

	/**
	 * Morada do aluno.
	 */
	private String address;

	/**
	 * Contacto do aluno.
	 */
	private String contact;

	/**
	 * Saldo do carto.
	 */
	private int balance;

	/**
	 * Estrutura de dados que guarda os movimentos de conta associados a um
	 * carto.
	 */
	private List<Transaction> transactions;

	/**
	 * Estrutura de dados que guarda os favoritos associados a um carto.
	 */
	private OrderedDictionary<String, Product> favouriteProducts;

	/**
	 * Construtor da classe CardClass.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param studentName
	 *            Nome do aluno.
	 * @param balance
	 *            Saldo do carto.
	 * @param studentClass
	 *            Turma do aluno.
	 * @param address
	 *            Morada do aluno.
	 * @param contact
	 *            Contacto do aluno.
	 */
	public CardClass(int cardCode, String studentName, int balance,
			String studentClass, String address, String contact) {
		transactions = new DoublyLinkedList<Transaction>();
		this.cardCode = cardCode;
		this.studentName = studentName;
		this.balance = balance;
		this.studentClass = studentClass;
		this.address = address;
		this.contact = contact;
		this.favouriteProducts = new AVLTree<String, Product>();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getCode()
	 */
	@Override
	public int getCode() {
		return cardCode;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getName()
	 */
	@Override
	public String getName() {
		return studentName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getBalance()
	 */
	@Override
	public int getBalance() {
		return balance;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getStudentClass()
	 */
	@Override
	public String getStudentClass() {
		return studentClass;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getaddress()
	 */
	@Override
	public String getAddress() {
		return address;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getContact()
	 */
	@Override
	public String getContact() {
		return contact;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getFavouriteProducts()
	 */
	@Override
	public Product getFavouriteProduct(String productName)
			throws CardWithoutFavouritesException,
			FavouriteProductInexistenceException {

		if (favouriteProducts.isEmpty())
			throw new CardWithoutFavouritesException();
		Product favourite = favouriteProducts.find(productName.toLowerCase()); 
		if ( favourite == null)
			throw new FavouriteProductInexistenceException();

		return favourite;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.Card#getTransactions()
	 */
	@Override
	public Iterator<Transaction> getTransactions()
			throws CardWithoutTransactionsException {
		if (transactions.isEmpty())
			throw new CardWithoutTransactionsException();
		return transactions.iterator();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.CardWritable#increaseBalance(int)
	 */
	@Override
	public void increaseBalance(int amount) {
		assert amount >= 0; // Assumimos que o valor de amount  sempre
		// positivo.
		balance += amount;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.CardWritable#addTransaction(int, int)
	 */
	@Override
	public void addTransaction(Product product, int totalItems) {
		Transaction t = new TransactionClass(product.getCode(), totalItems,
				product.getPrice());
		transactions.addLast(t); // Optmos por inserir sempre  cauda.

		int amount = product.getPrice() * totalItems;
		balance -= amount;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.CardWritable#addFavourite(schoolSystem.Product)
	 */
	@Override
	public void addFavourite(Product product)
			throws FavouriteProductAlreadyExistsException {

		if (favouriteProducts.find(product.getName().toLowerCase()) != null)
			throw new FavouriteProductAlreadyExistsException();
		
		favouriteProducts.insert(product.getName().toLowerCase(), product);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.CardWritable#removeFavourite(java.lang.String)
	 */
	@Override
	public void removeFavourite(String productName)
			throws FavouriteProductInexistenceException {
		if (favouriteProducts.find(productName.toLowerCase()) == null)
			throw new FavouriteProductInexistenceException();
		favouriteProducts.remove(productName.toLowerCase());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.CardWritable#clearTransactions()
	 */
	@Override
	public void clearTransactions() {
		transactions = new DoublyLinkedList<Transaction>();
	}

	/**
	 * Devolve a descrio de um carto.
	 * 
	 * @return Uma <code>String</code> com:
	 *         <code>cdigoCartao nomeAluno turma<br> morada contacto saldo</code>
	 *         .
	 */
	@Override
	public String toString() {
		return cardCode + " " + studentName + " " + studentClass + "\n"
				+ address + " " + contact + " " + balance + "\n";
	}

	@Override
	public Iterator<Entry<String, Product>> getFavourites()
			throws CardWithoutFavouritesException {
		if (favouriteProducts.isEmpty())
			throw new CardWithoutFavouritesException();

		return favouriteProducts.iterator();
	}

}
