package schoolSystem;


import dataStructures.*;

/**
 * @author Ricardo Gaspar Nr. 35277
 * @author Hugo Antnio Nr. 34334 Turno P2 Docente Vasco Amaral Interface da
 *         Aplicao SchoolSystem.
 */

public interface SchoolSystem {

	/**
	 * Insere um novo carto no sistema.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param studentName
	 *            Nome do aluno.
	 * @param balance
	 *            Saldo inicial do carto.
	 * @param studentClass
	 *            Turma do aluno.
	 * @param address
	 *            Morada do aluno.
	 * @param contact
	 *            Contacto do aluno.
	 * @throws CardAlreadyExistsException
	 *             Caso j exista um carto com o mesmo cdigo.
	 */
	void insertCard(int cardCode, String studentName, int balance,
			String studentClass, String address, String contact)
			throws CardAlreadyExistsException;

	/**
	 * Remove um carto do sistema.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @throws CardInexistenceException
	 *             Caso no exista nenhum carto com o cdigo indicado.
	 */
	void removeCard(int cardCode) throws CardInexistenceException;

	/**
	 * Devolve um carto existente no sistema.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @return Carto com o cdigo indicado.
	 * @throws CardInexistenceException
	 *             Caso no exista nenhum carto com o cdigo indicado.
	 */
	Card getCard(int cardCode) throws CardInexistenceException;

	/**
	 * Carrega um carto dado o seu cdigo e o valor a carregar.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param amount
	 *            Valor a carregar.
	 * @throws CardInexistenceException
	 *             Caso no exista nenhum carto com o cdigo indicado.
	 */
	void increaseCardBalance(int cardCode, int amount)
			throws CardInexistenceException;

	/**
	 * Insere um produto no sistema.
	 * 
	 * @param productCode
	 *            Cdigo do produto.
	 * @param productName
	 *            Nome do produto.
	 * @param productPrice
	 *            Preo do produto.
	 * @throws ProductAlreadyExistsException
	 *             Caso j exista um produto no sistema com o mesmo cdigo.
	 * @throws ProductNameAlreadyExistsException
	 *             Caso j exista um produto no sistema com o mesmo nome.
	 */
	void insertProduct(int productCode, String productName, int productPrice)
			throws ProductAlreadyExistsException,
			ProductNameAlreadyExistsException;

	/**
	 * Devolve um produto existente no sistema.
	 * 
	 * @param productCode
	 *            Cdigo do produto.
	 * @return Produto com o cdigo indicado.
	 * @throws ProductInexistenceException
	 *             Caso o produto com o cdigo indicado no exista.
	 */
	Product getProduct(int productCode) throws ProductInexistenceException;

	/**
	 * Compra de um produto.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param productCode
	 *            Cdigo do produto.
	 * @param totalItems
	 *            Nmero de itens a comprar do produto indicado.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws ProductInexistenceException
	 *             Caso o produto com o cdigo indicado no exista.
	 * @throws InsufficientBalanceException
	 *             Caso o saldo do carto indicado no tenha saldo insuficiente
	 *             para a compra a realizar.
	 */
	void purchaseProduct(int cardCode, int productCode, int totalItems)
			throws CardInexistenceException, ProductInexistenceException,
			InsufficientBalanceException;

	/**
	 * Adiciona um produto aos favoritos de um carto.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param productCode
	 *            Cdigo do produto.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws ProductInexistenceException
	 *             Caso o produto com o cdigo indicado no exista.
	 * @throws FavouriteProductAlreadyExistsException
	 *             Caso o produto j exista nos favoritos do carto.
	 */
	void addFavouriteProduct(int cardCode, int productCode)
			throws CardInexistenceException, ProductInexistenceException,
			FavouriteProductAlreadyExistsException;

	/**
	 * Compra de um produto favorito existente no carto.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param productName
	 *            Nome do produto.
	 * @param totalItems
	 *            Nmero de itens a comprar do produto indicado.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws FavouriteProductInexistenceException
	 *             Caso o produto no exista nos favoritos do carto.
	 * @throws InsufficientBalanceException
	 *             Caso o saldo do carto indicado no tenha saldo insuficiente
	 *             para a compra a realizar.
	 */
	void purchaseFavouriteProduct(int cardCode, String productName,
			int totalItems) throws CardInexistenceException,
			FavouriteProductInexistenceException, InsufficientBalanceException;

	/**
	 * Remove um produto existente nos favoritos de um carto.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @param productName
	 *            Nome do produto.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws FavouriteProductInexistenceException
	 *             Caso o produto no exista nos favoritos do carto.
	 */
	void removeFavouriteProduct(int cardCode, String productName)
			throws CardInexistenceException,
			FavouriteProductInexistenceException;

	/**
	 * Lista os produtos favoritos do carto indicado.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @return Iterador de entries de produtos favoritos existentes no carto.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws CardWithoutFavouritesException
	 *             Caso o carto no tenha produtos favoritos.
	 */
	Iterator<Entry<String, Product>> listFavouriteProducts(int cardCode)
			throws CardInexistenceException, CardWithoutFavouritesException;

	/**
	 * Lista todos os movimentos do carto indicado.
	 * 
	 * @param cardCode
	 *            Cdigo do carto.
	 * @return Iterador dos movimentos realizados com o carto indicado.
	 * @throws CardInexistenceException
	 *             Caso o carto com o cdigo indicado no exista.
	 * @throws CardWithoutTransactionsException
	 *             Caso o carto no tenha produtos favoritos.
	 */
	Iterator<Transaction> listTransactions(int cardCode)
			throws CardInexistenceException, CardWithoutTransactionsException;

	/**
	 * Muda de ms do sistema. Faz com que o valor de vendas de todos os
	 * produtos seja inicializado e os movimentos de todos os cartes existentes
	 * sejam removidos.
	 */
	void changeMonth();

	/**
	 * Lista todos os produtos existentes no sistema.
	 * 
	 * @return Iterador de entries de produtos existentes no sistema.
	 * @throws ProductInexistenceException
	 *             Caso no existam produtos no sistema.
	 */
	Iterator<Entry<String, Product>> listProducts() throws ProductInexistenceException;
}
