/**
 * 
 */
package schoolSystem;

import java.io.Serializable;

import dataStructures.*;

/**
 * @author Ricardo Gaspar Nr. 35277
 * @author Hugo Antnio Nr. 34334 Turno P2 Docente Vasco Amaral
 * 
 *         Classe que implementa a interface Product.
 */
public class SchoolSystemClass implements SchoolSystem, Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Capacidade inicial da estrutura de dados que armazena os cartes.
	 */
	private static final int CARDS_SIZE = 2000;

	/**
	 * Capacidade inicial da estrutura de dados que armazena os produtos.
	 */
	private static final int PRODUCTS_SIZE = 15000;

	/**
	 * Cartes existentes no sistema.
	 */
	private Dictionary<Integer, CardWritable> cards;

	/**
	 * Produtos existentes no sistema.
	 */
	private Dictionary<Integer, ProductWritable> products;

	/**
	 * Lista organizada por ordem alfabtica de produtos.
	 */
	private OrderedDictionary<String, Product> productList;

	/**
	 * Construtor da classe SchoolSystemClass.
	 */
	public SchoolSystemClass() {

		cards = new SepChainHashTable<Integer, CardWritable>(CARDS_SIZE);
		products = new SepChainHashTable<Integer, ProductWritable>(
				PRODUCTS_SIZE);
		productList = new AVLTree<String, Product>();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#insertCard(int, java.lang.String, int,
	 * java.lang.String, java.lang.String, int)
	 */
	@Override
	public void insertCard(int cardCode, String studentName, int balance,
			String studentClass, String address, String contact)
			throws CardAlreadyExistsException {

		// Verificar existncia do carto
		if (cards.find(cardCode) == null)
			cards.insert(cardCode, new CardClass(cardCode, studentName,
					balance, studentClass, address, contact));
		else
			throw new CardAlreadyExistsException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#removeCard(int)
	 */
	@Override
	public void removeCard(int cardCode) throws CardInexistenceException {

		// Verificar existncia do carto
		if (cards.find(cardCode) == null)
			throw new CardInexistenceException();

		cards.remove(cardCode);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#getCard(int)
	 */
	@Override
	public Card getCard(int cardCode) throws CardInexistenceException {

		// Verificar existncia do carto
		CardWritable card = cards.find(cardCode);
		if (card == null)
			throw new CardInexistenceException();
		return card;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#increaseCardBalance(int, int)
	 */
	@Override
	public void increaseCardBalance(int cardCode, int amount)
			throws CardInexistenceException {

		// Verificar existncia do carto
		CardWritable card = cards.find(cardCode);
		if (card == null)
			throw new CardInexistenceException();
		// Garantir que o valor  positivo
		assert amount >= 0;
		card.increaseBalance(amount);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#insertProduct(int, java.lang.String, int)
	 */
	@Override
	public void insertProduct(int productCode, String productName,
			int productPrice) throws ProductAlreadyExistsException,
			ProductNameAlreadyExistsException {

		// Verificar existncia do produto (na tabela) com o cdigo dado
		if (products.find(productCode) != null)
			throw new ProductAlreadyExistsException();

		// Verificar existncia do produto (na rvore) com o nome dado
		if (productList.find(productName.toLowerCase()) != null) {
			throw new ProductNameAlreadyExistsException();
		}

		ProductWritable product = new ProductClass(productCode, productName,
				productPrice);
		// Insero do produto na tabela e na rvore
		products.insert(productCode, product);
		productList.insert(productName.toLowerCase(), product);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#getProduct(int)
	 */
	@Override
	public Product getProduct(int productCode)
			throws ProductInexistenceException {

		// Verificar existncia do produto
		ProductWritable product = products.find(productCode);
		if (product == null)
			throw new ProductInexistenceException();

		return product;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#purchaseProduct(int, int, int)
	 */
	@Override
	public void purchaseProduct(int cardCode, int productCode, int totalItems)
			throws CardInexistenceException, ProductInexistenceException,
			InsufficientBalanceException {

		CardWritable card = cards.find(cardCode);
		ProductWritable product = products.find(productCode);
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();
		// Verificar existncia do produto
		if (product == null)
			throw new ProductInexistenceException();
		// Garantir que o carto tem saldo suficiente
		if ((product.getPrice() * totalItems) > card.getBalance())
			throw new InsufficientBalanceException();

		card.addTransaction(product, totalItems);
		product.increaseTotalSales(totalItems);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#addFavouriteProduct(int, int)
	 */
	@Override
	public void addFavouriteProduct(int cardCode, int productCode)
			throws CardInexistenceException, ProductInexistenceException,
			FavouriteProductAlreadyExistsException {

		CardWritable card = cards.find(cardCode);
		Product product = products.find(productCode);
		// Verificar existncia do produto
		if (product == null)
			throw new ProductInexistenceException();
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();

		try {
			if (card.getFavouriteProduct(product.getName()) != null)
				throw new FavouriteProductAlreadyExistsException();
		} catch (CardWithoutFavouritesException e) {
			card.addFavourite(product);
		} catch (FavouriteProductInexistenceException e) {
			card.addFavourite(product);
		}

	}

	@Override
	public void purchaseFavouriteProduct(int cardCode, String productName,
			int totalItems) throws CardInexistenceException,
			FavouriteProductInexistenceException, InsufficientBalanceException {

		CardWritable card = cards.find(cardCode);
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();

		try {
			Product favourite = card.getFavouriteProduct(productName);
			// Verificar existncia do produto favorito na tabela dos produtos
			ProductWritable product = products.find(favourite.getCode());
			if (product != null) {
				// Garantir que o carto tem saldo suficiente
				if (card.getBalance() >= favourite.getPrice() * totalItems) {
					card.addTransaction(favourite, totalItems);
					product.increaseTotalSales(totalItems);
				} else
					throw new InsufficientBalanceException();
			} else
				// Caso em que o favorito no existe nos produtos
				throw new FavouriteProductInexistenceException();
		} catch (CardWithoutFavouritesException e) {
			throw new FavouriteProductInexistenceException();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#removeFavouriteProduct(int,
	 * java.lang.String)
	 */
	@Override
	public void removeFavouriteProduct(int cardCode, String productName)
			throws CardInexistenceException,
			FavouriteProductInexistenceException {

		CardWritable card = cards.find(cardCode);
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();

		card.removeFavourite(productName);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#listFavouriteProducts(int)
	 */
	@Override
	public Iterator<Entry<String, Product>> listFavouriteProducts(int cardCode)
			throws CardInexistenceException, CardWithoutFavouritesException {

		CardWritable card = cards.find(cardCode);
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();

		return card.getFavourites();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#listTransactions(int)
	 */
	@Override
	public Iterator<Transaction> listTransactions(int cardCode)
			throws CardInexistenceException, CardWithoutTransactionsException {

		CardWritable card = cards.find(cardCode);
		// Verificar existncia do carto
		if (card == null)
			throw new CardInexistenceException();

		return card.getTransactions();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#changeMonth()
	 */
	@Override
	public void changeMonth() {

		// Remove todos os movimentos de cada carto
		Iterator<Entry<Integer, CardWritable>> it = cards.iterator();
		while (it.hasNext()) {
			it.next().getValue().clearTransactions();
		}
		// Inicializa as vendas de todos os produtos
		Iterator<Entry<Integer, ProductWritable>> itp = products.iterator();
		while (itp.hasNext()) {
			itp.next().getValue().clearTotalSales();
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see schoolSystem.SchoolSystem#listProducts()
	 */
	@Override
	public Iterator<Entry<String, Product>> listProducts()
			throws ProductInexistenceException {
		// Verificar se a rvore est vazia
		if (productList.isEmpty())
			throw new ProductInexistenceException();

		return productList.iterator();
	}

}
